// content.js
(function () {
  'use strict';

  // 默认配置
  const defaultOptions = {
    telegramStyle: false,
    darkMode: false,
    backgroundImage: 'https://r2.yckj.me/backblaze/tgbj.jpg',
    rightAlignSelf: false,
    showUnreadBadge: false,
    hideActionsMenu: false,
    autoCheckReply: false,
    enableCodeCopy: false,
    enableDblClickReply: false
  };
  let options = { ...defaultOptions };

  // 样式模板
  function getBaseCSS() {
    return `
      ${options.hideActionsMenu ? `.c-message_actions__container { display: none !important; }` : ''}
    `;
  }
  function getDarkModeCSS() {
    return `
      body, .p-workspace, .p-client_container, .p-top_nav, .p-channel_sidebar, .p-ia__nav__container, .p-threads_view__header, .p-threads_flexpane, .p-flexpane_header {
        background-color: #1a1d21 !important;
        color: #d1d2d3 !important;
      }
      .p-channel_sidebar--iap1, .p-channel_sidebar__list {
        background: #1a1d21 !important;
      }
      .p-workspace__primary_view_body, .c-virtual_list__scroll_container {
        background-color: #222529 !important;
      }
      .c-message_kit__actions .c-message_kit__gutter .c-message_kit__gutter__right {
        background-color: #2b2e32 !important;
        border: 1px solid #3d3d3d !important;
      }
      .c-message_kit__message, .c-message_kit__blocks--rich_text, .c-mrkdwn__text, .c-message_attachment__text {
        color: #dcddde !important;
      }
      .c-message__sender .c-message__sender_button {
        color: #b9bbbe !important;
        font-weight: 500;
      }
      a, .c-link {
        color: #3e9fff !important;
      }
      .p-channel_sidebar__channel--selected, .p-channel_sidebar__link--selected {
        background-color: #3b3e44 !important;
      }
      .p-channel_sidebar__link {
        color: #d1d2d3 !important;
      }
      .p-channel_sidebar__link:hover {
        background-color: #2a2d31 !important;
      }
      .p-channel_sidebar__channel_name {
         color: #d1d2d3 !important;
      }
      .p-message_input, .c-texty_input, .ql-editor, .c-texty_input--rosetta {
        background: #2b2e32 !important;
        color: #dcddde !important;
        border-color: #40444b !important;
      }
      .c-mrkdwn__pre, code, .c-mrkdwn__code {
        background-color: #2f3136 !important;
        border: 1px solid #414347 !important;
        color: #b9bbbe !important;
      }
      .c-message--mention {
         background-color: rgba(251, 189, 8, 0.15) !important;
      }
      .c-message--highlighted, .c-message--focus {
        background-color: rgba(62, 159, 255, 0.1) !important;
      }
      .p-workspace__primary_view_footer--float {
        background: #222529 !important;
        box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.6) !important;
      }
      .c-message_kit__sender {
        font-weight: 600 !important;
        color: #c97c38 !important;
      }
      .c-message__sender .c-message__sender_button {
        color: #c97c38 !important;
      }
    `;
  }
  function getTelegramCSS() {
    return `
      .p-file_drag_drop__container .c-scrollbar__hider {
        background-image: url('${options.backgroundImage}') !important;
        background-size: cover !important;
        background-repeat: repeat !important;
        background-attachment: fixed !important;
      }
      .c-message_kit__message {
        background: transparent !important;
        padding: 8px 0px !important;
        margin: 3px 5px !important;
        max-width: 500px !important;
        width: fit-content !important;
      }
      .c-message_kit__actions .c-message_kit__gutter .c-message_kit__gutter__right {
        background-color: rgba(255, 255, 255, 0.96) !important;
        border-radius: 15px !important;
        margin: -12px -8px -12px 0px !important;
        padding: 8px 12px !important;
        box-shadow: 0 1px 4px rgba(0, 0, 0, 0.08);
        max-width: 500px !important;
      }
      .c-message_kit__gutter__left {
        margin-top: -6px;
      }
      .c-base_icon__width_only_container {
        border-radius: 50%;
        box-shadow: 0 1px 4px rgba(0, 0, 0, 0.18);
      }
      .resize-triggers {
        position: relative !important;
        min-width: 350px !important;
      }
      .c-message_kit__attachments {
        min-width: 350px !important;
      }
      .p-message_pane_message__message .c-message_kit__gutter, .p-message_pane_message__message .c-message_kit__tombstone {
        padding: 4px 14px;
      }
      .c-message_kit__sender {
        font-weight: 600 !important;
        color: #1a73e8 !important;
      }
      .c-message__sender .c-message__sender_button {
        color: #1a73e8 !important;
      }
      .c-virtual_list__item {
        border-bottom: none !important;
      }
      .c-message--highlighted {
        background-color: #e6f0ff !important;
      }
      .c-virtual_list__scroll_container {
        background-color: transparent !important;
      }
      .p-workspace__primary_view_footer--float {
        background: transparent !important;
        margin-top: 0px;
        padding-top: 10px;
        box-shadow: 0px 0px 10px rgba(0, 0, 0, 0.37);
      }
      .c-message_kit__blocks--rich_text {
        margin-bottom: 0px;
      }
      .c-message_kit__thread_replies.c-message__reply_bar {
        margin-top: 0px;
      }
      .c-message_kit__reaction_bar {
        margin-top: 4px;
        margin-bottom: 0px;
      }
      .c-message_kit__labels__label {
        margin-bottom: 4px;
      }
    `;
  }

  // 注入样式
  function injectStyles() {
    // 移除旧样式
    document.getElementById('slack-enhance-style')?.remove();
    document.getElementById('slack-mh-telegram-style')?.remove();
    document.getElementById('slack-mh-dark-mode-style')?.remove();
    // 基础样式
    const style = document.createElement('style');
    style.id = 'slack-enhance-style';
    style.innerText = getBaseCSS();
    document.head.appendChild(style);
    // Telegram
    if (options.telegramStyle) {
      const tgStyle = document.createElement('style');
      tgStyle.id = 'slack-mh-telegram-style';
      tgStyle.innerText = getTelegramCSS();
      document.head.appendChild(tgStyle);
    }
    // 暗黑
    if (options.darkMode) {
      const dmStyle = document.createElement('style');
      dmStyle.id = 'slack-mh-dark-mode-style';
      dmStyle.innerText = getDarkModeCSS();
      document.head.appendChild(dmStyle);
    }
  }

  // 自动勾选 checkbox
  function setupAutoCheckReply() {
    if (options.autoCheckReply) {
      if (!window._slackAutoCheckObserver) {
        let lastCheckboxHandled = null;
        window._slackAutoCheckObserver = new MutationObserver(() => {
          // 找所有带有“同时发送至”的 label
          const labels = document.querySelectorAll('label.c-label');
          labels.forEach(label => {
              if (label.innerText.includes("同时发送至")) {
                  const checkbox = label.querySelector('input[type="checkbox"]');
                  if (checkbox && !checkbox.checked && checkbox !== lastCheckboxHandled) {
                      checkbox.click();
                      lastCheckboxHandled = checkbox;
                  }
              }
          });
        });
        window._slackAutoCheckObserver.observe(document.body, { childList: true, subtree: true });
      }
    } else {
      if (window._slackAutoCheckObserver) {
        window._slackAutoCheckObserver.disconnect();
        window._slackAutoCheckObserver = null;
      }
    }
  }

  // 高亮自己消息
  function highlightMessages() {
    // 自动获取当前登录用户名
    let myName = null;
    const userMenu = document.querySelector('[data-qa="user-button"]');
    if (userMenu) {
      myName = userMenu.getAttribute('aria-label') || userMenu.innerText;
    }
    let targetSenderName = myName ? myName.replace('用户：', '').trim() : '';
    // 高亮色根据模式切换
    let highlightColor = null;
    if (options.darkMode) {
      highlightColor = "#486993";
    } else if (options.telegramStyle) {
      highlightColor = "#e7fecc";
    } else {
      highlightColor = "#e7fecc";
    }
    document.querySelectorAll('[data-qa="message_content"]').forEach(messageContent => {
      const senderNameElement = messageContent.querySelector('button[data-qa="message_sender_name"]');
      const senderNameMatches = senderNameElement && senderNameElement.innerText.trim() === targetSenderName;
      if (senderNameMatches) {
        const gutter = messageContent.closest('.c-message_kit__gutter');
        const messageEl = gutter?.querySelector('.c-message_kit__gutter__right');
        if (messageEl && !messageEl.dataset.highlighted && highlightColor) {
          messageEl.style.setProperty('background-color', highlightColor, 'important');
          messageEl.dataset.highlighted = "true";
        }
        if (options.rightAlignSelf) {
          const wrapper = gutter.closest('.c-message_kit__message');
          if (wrapper) {
            wrapper.style.display = "flex";
            wrapper.style.justifyContent = "flex-end";
            wrapper.style.float = "right";
            wrapper.style.maxWidth = "500px";
          }
          if (gutter) {
            gutter.style.display = "flex";
            gutter.style.flexDirection = "row-reverse";
            gutter.style.alignItems = "flex-start";
            gutter.style.gap = "18px";
          }
        }
      }
    });
  }

  // 代码块复制
  function setupCodeCopy() {
    if (options.enableCodeCopy) {
      if (!window._slackCodeCopyObserver) {
        window._slackCodeCopyObserver = new MutationObserver(mutations => {
          mutations.forEach(mutation => {
            if (mutation.addedNodes.length > 0) {
              const pres = mutation.target.querySelectorAll('.c-mrkdwn__pre');
              pres.forEach(pre => {
                if (pre.querySelector('button.copy-btn')) return;
                pre.style.position = 'relative';
                pre.addEventListener('mouseenter', () => {
                  const btn = pre.querySelector('button.copy-btn');
                  if (btn) btn.style.display = 'block';
                });
                pre.addEventListener('mouseleave', () => {
                  const btn = pre.querySelector('button.copy-btn');
                  if (btn) btn.style.display = 'none';
                });
                const copyButton = document.createElement('button');
                copyButton.innerText = '复制';
                copyButton.className = 'copy-btn c-button c-button--primary c-button--small';
                Object.assign(copyButton.style, {
                  position: 'absolute',
                  right: '0',
                  top: '0',
                  display: 'none',
                  zIndex: 10
                });
                pre.appendChild(copyButton);
                copyButton.addEventListener('click', () => {
                  const code = Array.from(pre.childNodes)
                    .filter(node => node !== copyButton)
                    .map(node => node.textContent)
                    .join('')
                    .trim();
                  navigator.clipboard.writeText(code);
                  const prompt = document.createElement('div');
                  prompt.innerText = '复制成功!';
                  Object.assign(prompt.style, {
                    position: 'fixed',
                    top: '50px',
                    right: '50%',
                    transform: 'translateX(50%)',
                    padding: '10px 20px',
                    background: 'grey',
                    borderRadius: '4px',
                    zIndex: 222,
                    color: 'white',
                    fontWeight: 'bold'
                  });
                  document.body.appendChild(prompt);
                  setTimeout(() => {
                    document.body.removeChild(prompt);
                  }, 2000);
                });
              });
            }
          });
        });
        window._slackCodeCopyObserver.observe(document.body, { childList: true, subtree: true });
      }
    } else {
      if (window._slackCodeCopyObserver) {
        window._slackCodeCopyObserver.disconnect();
        window._slackCodeCopyObserver = null;
      }
    }
  }

  // 双击气泡边缘打开回复列表
  function setupDblClickReply() {
    if (options.enableDblClickReply) {
      if (!window._slackDblClickReplyObserver) {
        const observeMessages = () => {
          const observer = new MutationObserver(() => {
            const messages = document.querySelectorAll('.c-message_kit__message');
            messages.forEach(msg => {
              const gutter = msg.querySelector('.c-message_kit__gutter__right');
              if (gutter && !gutter.dataset.dblclickListenerAttached) {
                gutter.dataset.dblclickListenerAttached = 'true';
                gutter.addEventListener('dblclick', (e) => {
                  if (e.target !== gutter) return;
                  const threadButton = msg.querySelector('[data-qa="start_thread"]');
                  if (threadButton) {
                    threadButton.click();
                  }
                });
              }
            });
          });
          observer.observe(document.body, { childList: true, subtree: true });
          window._slackDblClickReplyObserver = observer;
        };
        const waitForSlack = setInterval(() => {
          if (document.querySelector('.c-message_kit__message')) {
            clearInterval(waitForSlack);
            observeMessages();
          }
        }, 1000);
      }
    } else {
      if (window._slackDblClickReplyObserver) {
        window._slackDblClickReplyObserver.disconnect();
        window._slackDblClickReplyObserver = null;
      }
    }
  }

  // 频道未读红标“新”
  function setupUnreadBadge() {
    if (options.showUnreadBadge) {
      if (!window._slackUnreadBadgeObserver) {
        const style = document.createElement('style');
        style.id = 'slack-mh-unread-badge-style';
        style.textContent = `
          .tm-dot {
            display: inline-block;
            background-color: red;
            color: white;
            font-size: 10px;
            font-weight: bold;
            border-radius: 6px;
            padding: 0 4px;
            margin-left: 6px;
            line-height: 14px;
            vertical-align: middle;
            font-family: sans-serif;
            user-select: none;
          }
        `;
        document.head.appendChild(style);
        function enhanceUnreadChannels() {
          const channels = document.querySelectorAll('.p-channel_sidebar__channel');
          channels.forEach(channel => {
            const parent = channel.closest('[role="treeitem"]');
            if (!parent) return;
            const isUnread = channel.classList.contains('p-channel_sidebar__channel--unread');
            const nameSpan = channel.querySelector('.p-channel_sidebar__name');
            if (!nameSpan) return;
            const existingDot = nameSpan.querySelector('.tm-dot');
            if (existingDot) existingDot.remove();
            if (isUnread) {
              const dot = document.createElement('span');
              dot.className = 'tm-dot';
              dot.textContent = '新';
              nameSpan.appendChild(dot);
            }
          });
        }
        setTimeout(enhanceUnreadChannels, 1000);
        let timeout;
        window._slackUnreadBadgeObserver = new MutationObserver(() => {
          clearTimeout(timeout);
          timeout = setTimeout(enhanceUnreadChannels, 300);
        });
        window._slackUnreadBadgeObserver.observe(document.body, { childList: true, subtree: true });
      }
    } else {
      document.getElementById('slack-mh-unread-badge-style')?.remove();
      if (window._slackUnreadBadgeObserver) {
        window._slackUnreadBadgeObserver.disconnect();
        window._slackUnreadBadgeObserver = null;
      }
    }
  }

  // 主应用函数
  function applySlackEnhanceOptions() {
    injectStyles();
    setupAutoCheckReply();
    setupCodeCopy();
    setupDblClickReply();
    setupUnreadBadge();
    highlightMessages();
    if (!window._slackHighlightObserver) {
      window._slackHighlightObserver = new MutationObserver(highlightMessages);
      window._slackHighlightObserver.observe(document.body, { childList: true, subtree: true });
    }
  }

  // 读取配置并应用
  function loadAndApplyOptions() {
    chrome.storage.sync.get(defaultOptions, (items) => {
      options = { ...defaultOptions, ...items };
      applySlackEnhanceOptions();
    });
  }

  // 监听 storage 变化
  chrome.storage.onChanged.addListener((changes, area) => {
    if (area === 'sync') {
      loadAndApplyOptions();
    }
  });

  // 监听 popup 的刷新消息
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if (msg && msg.type === 'REFRESH_SLACK_OPTIMIZER') {
      loadAndApplyOptions();
    }
  });

  // 首次加载
  loadAndApplyOptions();

})(); 